//*************************************************************************************************
//
//	Description:
//		admanager_advert.fx - Shader used to display downloaded adverts from massive, IGX etc. Very similar to a stripped down
//													version of basic.fx, but with extra lighting code added to ensure the result is never too dark or too
//													light.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     11/06/2009  0.1           Created
//	<TABLE>
//
//*************************************************************************************************

#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// Modify this scaling factor to change the brightness of the advert textures before they're plugged into the lighting

// PS3
#if defined( _PS3_ )
#define GLOBAL_BRIGHTNESS_SCALE	0.5f

// 360
#elif defined( _XBOX )
#define GLOBAL_BRIGHTNESS_SCALE	0.4f

// PC
#else
#define GLOBAL_BRIGHTNESS_SCALE	0.5f
#endif

// Modify these to change the specular power/factor for all adverts
#define GLOBAL_SPECULAR_FACTOR	0.1f
#define GLOBAL_SPECULAR_POWER		32.0f

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#endif


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//
#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
>;
#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS


#if defined( _3DSMAX_ )
#define GetLightDirection( _input ) _input.lightVector
#define GetLightColour( _input ) lightColour
#elif 		defined( _LIGHTING_D_ )		|| 	defined( _LIGHTING_DD_ )	|| 	defined( _LIGHTING_DDD_ ) || defined( _LIGHTING_DP_ )	||	defined( _LIGHTING_DS_ )	|| 	defined( _LIGHTING_DPS_ ) || defined( _LIGHTING_DPP_ )	|| 	defined( _LIGHTING_DSS_ ) || 	defined( _LIGHTING_DDP_ ) || defined( _LIGHTING_DDS_ ) 
#define GetLightDirection( _input ) lightDirection0
#define GetLightColour( _input ) lightColour0
#else
#define GetLightDirection( _input ) float3(0.0f,1.0f,0.0f)
#define GetLightColour( _input ) float4(1.0f, 1.0f, 1.0f, 1.0f)
#endif


//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float3 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD2;													// Vertex alpha
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
	float3 normal   : NORMAL;															// Object space normal
};


struct SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// Texture coords
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;												// Ambient lighting
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)
	float4 vertColour	: TEXCOORD4;												// Original vertex colour

	DECLARE_LIGHTING_INTERPOLATORS_VS( 5 )
};


struct SHADOWGEN_VSOUTPUT
{
	float4 position			: POSITION;
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 shadowCoord	: TEXCOORD1;
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT AdvertVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.vertColour = _output.colour;
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	_output.normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );

	// We always ambient light these adverts as if they're facing upwards, to ensure they are visible even in shadow
	float3 normal = float3( 0.0f, 1.0f, 0.0f );

	float globalSpecularFactorValue = GLOBAL_SPECULAR_FACTOR;

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	_output.colour = float4( 0.5, 0.5, 0.5, 1.0f );

	return _output;
}



SHADOWGEN_VSOUTPUT AdvertTranslucentShadowGenVertexShader( SHADOWGEN_VSINPUT _input )
{
	SHADOWGEN_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;
	_output.colour	 = _input.colour;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Clamp geometry that is behind the camera to the near plane so that it still renders.
	// We use an orthogonal projection, so doing this will not distort the shadow caster.
	CLAMP_SHADOW_Z;

	OUTPUT_SHADOW_COORDS;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

struct PSINPUT
{
	float4 colour			: TEXCOORD3;		// Vertex colour
	float2 texCoord		: TEXCOORD0;		// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;		// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2;		// Eye vector (world space)
	float4 vertColour	: TEXCOORD4;		// Original vertex colour

	DECLARE_LIGHTING_INTERPOLATORS_PS( 5 )
};

#else

// Input structure
struct PSINPUT
{
	float4 colour		: TEXCOORD3_centroid;		// Vertex colour
	float2 texCoord	: TEXCOORD0;						// UV coords for texture channel 0
	float3 normal		: TEXCOORD1_centroid;		// Normal vector (world space) + fresnel coefficient
	float4 eye			: TEXCOORD2_centroid;		// Eye vector (world space)
	float4 vertColour	: TEXCOORD4_centroid;	// Original vertex colour

	DECLARE_LIGHTING_INTERPOLATORS_PS( 5 )
	DECLARE_SHADOW_PS_INPUTS
};

#endif	// defined( _3DSMAX_ )


struct SHADOWGEN_PSINPUT
{
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 shadowCoord	: TEXCOORD1;
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT AdvertFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	float4 specularTexColour = float4( 1.0f, 1.0f, 1.0f, 0.0f );

	float globalSpecularFactorValue = GLOBAL_SPECULAR_FACTOR;
	float minSpecPowerValue = GLOBAL_SPECULAR_POWER;
	float maxSpecPowerValue = GLOBAL_SPECULAR_POWER;

	// Factor vertex colour and global scale into the diffuse colour
	diffuseTexColour *= _input.vertColour;
	diffuseTexColour.rgb *= GLOBAL_BRIGHTNESS_SCALE;

  // Normalise interpolated vectors
	float3 normal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );

	// Calculate base colour (diffuse * ambient lighting)
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour, _input.eye.xyz )

	// Wipe the results of the lighting out, we're only interested in the shadow value
	accumulatedColour = diffuseTexColour * _input.colour;

	// Apply reference direct lighting
#if defined( _3DSMAX_ )
	float PSL_attenuation = 1.0f;
#endif
	float3 tomtest_L = -GetLightDirection(_input);
	accumulatedColour += float4( 0.5f, 0.5f, 0.5f, 0.0f ) * diffuseTexColour * (1.0f - GLOBAL_SPECULAR_FACTOR ) * max( CalculateHalfLambertDiffuseLighting( normal, tomtest_L ), 0.5f ) * PSL_attenuation;
	accumulatedColour += float4( 0.5f, 0.5f, 0.5f, 0.0f ) * specularTexColour * GLOBAL_SPECULAR_FACTOR * CalculateSpecularLighting( normal, tomtest_L, eye, GLOBAL_SPECULAR_POWER ) * PSL_attenuation;

	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateOutputPixel( accumulatedColour );

	return _output;
}


//
// Low Detail Shaders
//


struct VSOUTPUT_LD
{
	float4 position		: POSITION;			// View-coords position
	float4 colour			: TEXCOORD1;		// Vertex lighting colour
	float2 texCoord		: TEXCOORD0;		// UV coords for texture channel 0
	float4 vertColour	: TEXCOORD2;		// Original vertex colour
};


VSOUTPUT_LD AdvertLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// get normal in world space and do lighting
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.vertColour = _output.colour;
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	// Ensure the "lighting alpha" is 1.0; we'll factor in the vertex alpha using the original vert colour
	_output.colour.a = 1.0f;

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT AdvertLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// Factor vertex colour and global scale into the diffuse colour
	diffuseTexColour *= _input.vertColour;
	diffuseTexColour.rgb *= GLOBAL_BRIGHTNESS_SCALE;

	// Apply lighting to base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );

	return _output;
}


REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT AdvertTranslucentShadowGenFragmentShader( SHADOWGEN_PSINPUT _input )
{
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );
	output.Colour.a = diffuseTexColour.a * _input.colour.a;
	SHADOWMAP_PS_ALPHATEST( output.Colour.a, 0.25f );

	return output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Advert
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	bool disableGlobalMipBias = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Advert";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "_Advert_LowDetail";
	int    lowDetailDeferredID = 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx AdvertVertexShader();
		PixelShader = compile sce_fp_rsx AdvertFragmentShader();
#else
		VertexShader = compile vs_3_0 AdvertVertexShader();
		PixelShader = compile ps_3_0 AdvertFragmentShader();
#endif
	}
}



technique Advert_Translucent
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	bool disableGlobalMipBias = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "Advert";
	int    normalDeferredID			= 2;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_Advert_Translucent_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string lowDetailBehaviour		= "ERMB_RENDER";
	string lowDetailTechnique		= "_Advert_LowDetail";
	int    lowDetailDeferredID	= 2;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 AdvertVertexShader();
		PixelShader = compile ps_3_0 AdvertFragmentShader();
#endif
	}
}


technique _Advert_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	bool disableGlobalMipBias = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "_Advert_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx AdvertLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx AdvertLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 AdvertLowDetailVertexShader();
		PixelShader = compile ps_3_0 AdvertLowDetailFragmentShader();
#endif
	}
}


technique _Advert_Translucent_ShadowGen
<
	 bool disableGlobalMipBias = true;
>
{

	pass Pass0
	{
			SHADOWMAP_STATES_ALPHATEST( 0x40 )
#if defined (_PS3_)
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx AdvertTranslucentShadowGenVertexShader();
			PixelShader = compile sce_fp_rsx AdvertTranslucentShadowGenFragmentShader();
#elif defined(_XBOX)
			VertexShader = compile vs_3_0 AdvertTranslucentShadowGenVertexShader();
			PixelShader = compile ps_3_0 AdvertTranslucentShadowGenFragmentShader();
#else
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 AdvertTranslucentShadowGenVertexShader();
			PixelShader = compile ps_3_0 AdvertTranslucentShadowGenFragmentShader();
#endif
	}
}
